import { useState, useEffect } from 'react';
import { usePage, router } from '@inertiajs/react';
import { PageTemplate } from '@/components/page-template';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { DatePicker } from '@/components/ui/date-picker';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { ArrowLeft, Users, Eye, MessageSquare, Smartphone, Laptop, Globe } from 'lucide-react';
import { useTranslation } from 'react-i18next';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';

// Import chart components
import {
  LineChart,
  Line,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
} from 'recharts';

interface Analytics {
  pageViews: number;
  uniqueVisitors: number;
  dailyPageViews: Array<{
    date: string;
    views: number;
  }>;
  referrers: Array<{
    referrer: string;
    count: number;
  }>;
  devices: Array<{
    device_type: string;
    count: number;
  }>;
  platforms: Array<{
    platform_name: string;
    count: number;
  }>;
  browsers: Array<{
    browser_name: string;
    count: number;
  }>;
  startDate: string;
  endDate: string;
}

interface BioLink {
  id: number;
  slug: string;
  name: string;
  link_type: string;
}

export default function BioLinkAnalytics() {
  const { t } = useTranslation();
  const { bioLink, analytics, filters } = usePage().props as {
    bioLink: BioLink;
    analytics: Analytics;
    filters: {
      start_date?: string;
      end_date?: string;
    };
  };

  const [startDate, setStartDate] = useState<Date | undefined>(
    filters.start_date ? new Date(filters.start_date) : new Date(analytics.startDate)
  );
  const [endDate, setEndDate] = useState<Date | undefined>(
    filters.end_date ? new Date(filters.end_date) : new Date(analytics.endDate)
  );

  const [isLoading, setIsLoading] = useState(false);

  const handleDateChange = () => {
    if (!startDate || !endDate) return;
    
    setIsLoading(true);
    router.get(
      route('link-bio-builder.analytics', bioLink.id),
      {
        start_date: startDate.toISOString().split('T')[0],
        end_date: endDate.toISOString().split('T')[0],
      },
      { 
        preserveState: true,
        onFinish: () => setIsLoading(false)
      }
    );
  };

  // Auto-refresh every 5 minutes for real-time data
  useEffect(() => {
    const interval = setInterval(() => {
      if (!isLoading) {
        router.reload({ only: ['analytics'] });
      }
    }, 300000); // 5 minutes

    return () => clearInterval(interval);
  }, [isLoading]);

  // Chart colors
  const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884d8'];

  // Format date for chart display
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return `${date.getMonth() + 1}/${date.getDate()}`;
  };

  // Prepare data for device type pie chart
  const deviceData = analytics.devices?.length > 0 ? analytics.devices.map((item, index) => ({
    name: item.device_type || 'Unknown',
    value: item.count,
  })) : [{ name: 'No Data', value: 1 }];

  // Prepare data for referrers bar chart
  const referrerData = analytics.referrers?.length > 0 ? analytics.referrers.map((item) => ({
    name: item.referrer || 'Direct',
    value: item.count,
  })) : [{ name: 'No Data', value: 1 }];
  
  // Prepare data for platforms pie chart
  const platformData = analytics.platforms?.length > 0 ? analytics.platforms.map((item) => ({
    name: item.platform_name || 'Unknown',
    value: item.count,
  })) : [{ name: 'No Data', value: 1 }];
  
  // Prepare data for browsers bar chart
  const browserData = analytics.browsers?.length > 0 ? analytics.browsers.map((item) => ({
    name: item.browser_name || 'Unknown',
    value: item.count,
  })) : [{ name: 'No Data', value: 1 }];

  // Define page actions
  const pageActions = [
    {
      label: t('Back to Bio Links'),
      icon: <ArrowLeft className="h-4 w-4 mr-2" />,
      variant: 'outline' as const,
      onClick: () => router.get(route('link-bio-builder.index')),
    },
  ];

  const breadcrumbs = [
    { title: t('Dashboard'), href: route('dashboard') },
    { title: t('Bio Link Builder'), href: route('link-bio-builder.index') },
    { title: bioLink?.name || 'Bio Link', href: route('link-bio-builder.edit', bioLink?.id) },
    { title: t('Analytics') },
  ];

  return (
    <PageTemplate
      title={`${bioLink?.name} - ${t('Analytics')}`}
      url={route('link-bio-builder.analytics', bioLink?.id)}
      actions={pageActions}
      breadcrumbs={breadcrumbs}
    >
      <div className="space-y-6">
        {/* Date Range Selector */}
        <Card>
          <CardHeader>
            <CardTitle className="text-sm">{t('Date Range')}</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-col sm:flex-row gap-4 items-end">
              <div className="space-y-2">
                <Label>{t('Start Date')}</Label>
                <DatePicker
                  selected={startDate}
                  onSelect={setStartDate}
                  onChange={(date) => setStartDate(date)}
                />
              </div>
              <div className="space-y-2">
                <Label>{t('End Date')}</Label>
                <DatePicker
                  selected={endDate}
                  onSelect={setEndDate}
                  onChange={(date) => setEndDate(date)}
                />
              </div>
              <Button onClick={handleDateChange} disabled={isLoading}>
                {isLoading ? t('Loading...') : t('Apply')}
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Key Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">{t('Page Views')}</p>
                  <h3 className="mt-2 text-2xl font-bold">{analytics.pageViews}</h3>
                </div>
                <div className="rounded-full bg-blue-100 p-3 dark:bg-blue-900">
                  <Eye className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">{t('Unique Visitors')}</p>
                  <h3 className="mt-2 text-2xl font-bold">{analytics.uniqueVisitors}</h3>
                </div>
                <div className="rounded-full bg-purple-100 p-3 dark:bg-purple-900">
                  <Users className="h-5 w-5 text-purple-600 dark:text-purple-400" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-muted-foreground">{t('Link Type')}</p>
                  <h3 className="mt-2 text-lg font-bold">
                    <Badge variant="outline" className="capitalize">
                      {bioLink.link_type}
                    </Badge>
                  </h3>
                </div>
                <div className="rounded-full bg-green-100 p-3 dark:bg-green-900">
                  <Globe className="h-5 w-5 text-green-600 dark:text-green-400" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Charts */}
        <Tabs defaultValue="traffic">
          <TabsList className="mb-4 w-full justify-start overflow-x-auto">
            <TabsTrigger value="traffic">{t('Traffic')}</TabsTrigger>
            <TabsTrigger value="devices">{t('Devices')}</TabsTrigger>
            <TabsTrigger value="platforms">{t('Platforms')}</TabsTrigger>
            <TabsTrigger value="browsers">{t('Browsers')}</TabsTrigger>
            <TabsTrigger value="referrers">{t('Referrers')}</TabsTrigger>
          </TabsList>
          
          <TabsContent value="traffic">
            <Card>
              <CardHeader>
                <CardTitle className="text-sm">{t('Daily Page Views')}</CardTitle>
                <CardDescription className="text-xs">
                  {t('Page views over the selected time period')}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="h-[300px]">
                  <ResponsiveContainer width="100%" height="100%">
                    <LineChart
                      data={analytics.dailyPageViews || []}
                      margin={{ top: 5, right: 30, left: 20, bottom: 5 }}
                    >
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis
                        dataKey="date"
                        tickFormatter={formatDate}
                      />
                      <YAxis />
                      <Tooltip />
                      <Legend />
                      <Line
                        type="monotone"
                        dataKey="views"
                        name="Page Views"
                        stroke="#8884d8"
                        activeDot={{ r: 8 }}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
          
          <TabsContent value="devices">
            <Card>
              <CardHeader>
                <CardTitle className="text-sm">{t('Device Types')}</CardTitle>
                <CardDescription className="text-xs">
                  {t('Breakdown of visitors by device type')}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="h-[300px]">
                  <ResponsiveContainer width="100%" height="100%">
                    <PieChart>
                      <Pie
                        data={deviceData}
                        cx="50%"
                        cy="50%"
                        labelLine={false}
                        outerRadius={100}
                        fill="#8884d8"
                        dataKey="value"
                        label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                      >
                        {deviceData.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Pie>
                      <Tooltip />
                      <Legend />
                    </PieChart>
                  </ResponsiveContainer>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
          
          <TabsContent value="platforms">
            <Card>
              <CardHeader>
                <CardTitle className="text-sm">{t('Operating Systems')}</CardTitle>
                <CardDescription className="text-xs">
                  {t('Breakdown of visitors by operating system')}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="h-[300px]">
                  <ResponsiveContainer width="100%" height="100%">
                    <PieChart>
                      <Pie
                        data={platformData}
                        cx="50%"
                        cy="50%"
                        labelLine={false}
                        outerRadius={100}
                        fill="#8884d8"
                        dataKey="value"
                        label={({ name, percent }) => `${name}: ${(percent * 100).toFixed(0)}%`}
                      >
                        {platformData.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Pie>
                      <Tooltip />
                      <Legend />
                    </PieChart>
                  </ResponsiveContainer>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
          
          <TabsContent value="browsers">
            <Card>
              <CardHeader>
                <CardTitle className="text-sm">{t('Web Browsers')}</CardTitle>
                <CardDescription className="text-xs">
                  {t('Breakdown of visitors by web browser')}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="h-[300px]">
                  <ResponsiveContainer width="100%" height="100%">
                    <BarChart
                      data={browserData}
                      margin={{ top: 5, right: 30, left: 20, bottom: 5 }}
                    >
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="name" />
                      <YAxis />
                      <Tooltip />
                      <Legend />
                      <Bar dataKey="value" name="Visits" fill="#8884d8">
                        {browserData.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Bar>
                    </BarChart>
                  </ResponsiveContainer>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
          
          <TabsContent value="referrers">
            <Card>
              <CardHeader>
                <CardTitle className="text-sm">{t('Top Referrers')}</CardTitle>
                <CardDescription className="text-xs">
                  {t('Sources of traffic to your bio link')}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="h-[300px]">
                  <ResponsiveContainer width="100%" height="100%">
                    <BarChart
                      data={referrerData}
                      margin={{ top: 5, right: 30, left: 20, bottom: 5 }}
                    >
                      <CartesianGrid strokeDasharray="3 3" />
                      <XAxis dataKey="name" />
                      <YAxis />
                      <Tooltip />
                      <Legend />
                      <Bar dataKey="value" name="Visits" fill="#8884d8">
                        {referrerData.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                        ))}
                      </Bar>
                    </BarChart>
                  </ResponsiveContainer>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </PageTemplate>
  );
}