<?php

namespace App\Http\Controllers\Settings;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Services\StorageConfigService;

class SystemSettingsController extends Controller
{
    /**
     * Update the system settings.
     *
     * Handles system-wide configuration including:
     * - Language and localization settings
     * - Date/time formats and timezone
     * - Email verification requirements
     * - Landing page enable/disable toggle
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        try {
            // Different validation rules based on user type
            if (auth()->user()->type === 'company') {
                $validated = $request->validate([
                    'defaultLanguage' => 'required|string',
                    'dateFormat' => 'required|string',
                    'timeFormat' => 'required|string',
                    'calendarStartDay' => 'required|string',
                    'defaultTimezone' => 'required|string',
                ]);
            } else {
                $validated = $request->validate([
                    'defaultLanguage' => 'required|string',
                    'dateFormat' => 'required|string',
                    'timeFormat' => 'required|string',
                    'calendarStartDay' => 'required|string',
                    'defaultTimezone' => 'required|string',
                    'emailVerification' => 'boolean',
                    'landingPageEnabled' => 'boolean',
                    'registrationEnabled' => 'boolean',
                    'termsConditionsUrl' => 'nullable|url',
                ]);
            }

            foreach ($validated as $key => $value) {
                updateSetting($key, $value);
            }

            return redirect()->back()->with('success', __('System settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update system settings: :error', ['error' => $e->getMessage()]));
        }
    }
    
    /**
     * Update the brand settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateBrand(Request $request)
    {
        try {
            $validated = $request->validate([
                'settings' => 'required|array',
                'settings.logoDark' => 'nullable|string',
                'settings.logoLight' => 'nullable|string',
                'settings.favicon' => 'nullable|string',
                'settings.titleText' => 'nullable|string|max:255',
                'settings.footerText' => 'nullable|string|max:500',
                'settings.themeColor' => 'nullable|string|in:blue,green,purple,orange,red,custom',
                'settings.customColor' => 'nullable|string|regex:/^#[0-9A-Fa-f]{6}$/',
                'settings.sidebarVariant' => 'nullable|string|in:inset,floating,minimal',
                'settings.sidebarStyle' => 'nullable|string|in:plain,colored,gradient',
                'settings.layoutDirection' => 'nullable|string|in:left,right,ltr,rtl',
                'settings.themeMode' => 'nullable|string|in:light,dark,system',
                // Add file upload validation
                'logoDarkFile' => 'nullable|file|mimes:png,jpg,jpeg,gif,svg|max:2048',
                'logoLightFile' => 'nullable|file|mimes:png,jpg,jpeg,gif,svg|max:2048',
                'faviconFile' => 'nullable|file|mimes:png,jpg,jpeg,gif,svg,ico|max:2048',
            ]);

            $userId = auth()->id();
            $currentStorageType = getSetting('storage_type', 'local');
            $activeDisk = StorageConfigService::getActiveDisk();
            
            // Handle file uploads
            foreach (['logoDark', 'logoLight', 'favicon'] as $assetType) {
                $fileKey = $assetType . 'File';
                if ($request->hasFile($fileKey)) {
                    $file = $request->file($fileKey);
                    $extension = $file->getClientOriginalExtension();
                    $filename = $assetType . '_' . time() . '.' . $extension;
                    $path = 'brand/' . $filename;
                    
                    // Store file to current storage type
                    \Storage::disk($activeDisk)->put($path, file_get_contents($file));
                    
                    // Store with storage type prefix
                    $validated['settings'][$assetType] = $currentStorageType . ':' . $path;
                }
            }
            
            // Update settings in database
            foreach ($validated['settings'] as $key => $value) {
                // Convert layoutDirection from frontend format to backend format if needed
                if ($key === 'layoutDirection') {
                    $value = $value === 'left' ? 'ltr' : ($value === 'right' ? 'rtl' : $value);
                }
                
                // For brand assets, add storage type prefix if not already present
                if (in_array($key, ['logoDark', 'logoLight', 'favicon']) && $value && !str_contains($value, ':')) {
                    $value = $currentStorageType . ':' . $this->normalizeBrandAssetPath($value);
                }
                
                updateSetting($key, $value, $userId);
            }
            
            // Clear any cached settings to ensure fresh data on next request
            \Cache::forget('user_settings_' . $userId);
            \Cache::forget('global_settings');
            
            // Force refresh of settings in the response
            $refreshedSettings = settings($userId);

            // Share updated settings with Inertia
            Inertia::share('globalSettings', settings($userId));
            
            return redirect()->back()
                ->with('success', __('Brand settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update brand settings: :error', ['error' => $e->getMessage()]));
        }
    }
    
    /**
     * Normalize brand asset path to be storage-type agnostic
     */
    private function normalizeBrandAssetPath($path)
    {
        if (!$path) return $path;
        
        // If it's already a full URL, extract just the filename
        if (str_starts_with($path, 'http')) {
            $path = basename($path);
        }
        
        // Remove /storage/ prefix if present
        $path = str_replace('/storage/', '', $path);
        
        // Remove leading slash if present
        return ltrim($path, '/');
    }
    


    /**
     * Update the recaptcha settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateRecaptcha(Request $request)
    {
        try {
            $validated = $request->validate([
                'recaptchaEnabled' => 'boolean',
                'recaptchaVersion' => 'required|in:v2,v3',
                'recaptchaSiteKey' => 'required|string',
                'recaptchaSecretKey' => 'required|string',
            ]);
            
            foreach ($validated as $key => $value) {
                updateSetting($key, $value);
            }

            return redirect()->back()->with('success', __('ReCaptcha settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update ReCaptcha settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the chatgpt settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateChatgpt(Request $request)
    {
        try {
            $validated = $request->validate([
                'chatgptKey' => 'required|string',
                'chatgptModel' => 'required|string',
            ]);
            
            foreach ($validated as $key => $value) {
                updateSetting($key, $value);
            }

            return redirect()->back()->with('success', __('Chat GPT settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update Chat GPT settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the storage settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateStorage(Request $request)
    {
        try {
            $validated = $request->validate([
                'storage_type' => 'required|in:local,aws_s3,wasabi',
                'allowedFileTypes' => 'required|string',
                'maxUploadSize' => 'required|numeric|min:1',
                'awsAccessKeyId' => 'required_if:storage_type,aws_s3|string',
                'awsSecretAccessKey' => 'required_if:storage_type,aws_s3|string',
                'awsDefaultRegion' => 'required_if:storage_type,aws_s3|string',
                'awsBucket' => 'required_if:storage_type,aws_s3|string',
                'awsUrl' => 'required_if:storage_type,aws_s3|string',
                'awsEndpoint' => 'required_if:storage_type,aws_s3|string',
                'wasabiAccessKey' => 'required_if:storage_type,wasabi|string',
                'wasabiSecretKey' => 'required_if:storage_type,wasabi|string',
                'wasabiRegion' => 'required_if:storage_type,wasabi|string',
                'wasabiBucket' => 'required_if:storage_type,wasabi|string',
                'wasabiUrl' => 'required_if:storage_type,wasabi|string',
                'wasabiRoot' => 'required_if:storage_type,wasabi|string',
            ]);

            $userId = Auth::id();
            
            $settings = [
                'storage_type' => $validated['storage_type'],
                'storage_file_types' => $validated['allowedFileTypes'],
                'storage_max_upload_size' => $validated['maxUploadSize'],
            ];

            if ($validated['storage_type'] === 'aws_s3') {
                $settings['aws_access_key_id'] = $validated['awsAccessKeyId'];
                $settings['aws_secret_access_key'] = $validated['awsSecretAccessKey'];
                $settings['aws_default_region'] = $validated['awsDefaultRegion'];
                $settings['aws_bucket'] = $validated['awsBucket'];
                $settings['aws_url'] = $validated['awsUrl'];
                $settings['aws_endpoint'] = $validated['awsEndpoint'];
            }

            if ($validated['storage_type'] === 'wasabi') {
                $settings['wasabi_access_key'] = $validated['wasabiAccessKey'];
                $settings['wasabi_secret_key'] = $validated['wasabiSecretKey'];
                $settings['wasabi_region'] = $validated['wasabiRegion'];
                $settings['wasabi_bucket'] = $validated['wasabiBucket'];
                $settings['wasabi_url'] = $validated['wasabiUrl'];
                $settings['wasabi_root'] = $validated['wasabiRoot'];
            }
            
            foreach ($settings as $key => $value) {
                updateSetting($key, $value);
            }

            StorageConfigService::clearCache();

            return redirect()->back()->with('success', __('Storage settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update storage settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the cookie settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateCookie(Request $request)
    {
        try {
            $validated = $request->validate([
                'enableLogging' => 'required|boolean',
                'strictlyNecessaryCookies' => 'required|boolean',
                'cookieTitle' => 'required|string|max:255',
                'strictlyCookieTitle' => 'required|string|max:255',
                'cookieDescription' => 'required|string',
                'strictlyCookieDescription' => 'required|string',
                'contactUsDescription' => 'required|string',
                'contactUsUrl' => 'required|url',
            ]);
            
            foreach ($validated as $key => $value) {
                updateSetting($key, is_bool($value) ? ($value ? '1' : '0') : $value);
            }

            return redirect()->back()->with('success', __('Cookie settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update cookie settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the SEO settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateSeo(Request $request)
    {
        try {
            $validated = $request->validate([
                'metaKeywords' => 'required|string|max:255',
                'metaDescription' => 'required|string|max:160',
                'metaImage' => 'required|string',
            ]);
            
            foreach ($validated as $key => $value) {
                updateSetting($key, $value);
            }

            return redirect()->back()->with('success', __('SEO settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update SEO settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the Google Calendar settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateGoogleCalendar(Request $request)
    {
        try {
            $validated = $request->validate([
                'googleCalendarEnabled' => 'boolean',
                'googleCalendarId' => 'nullable|string|max:255',
                'googleCalendarJson' => 'nullable|file|mimes:json|max:2048',
            ]);

            $settings = [
                'googleCalendarEnabled' => $validated['googleCalendarEnabled'] ?? false,
                'googleCalendarId' => $validated['googleCalendarId'] ?? '',
            ];

            // Handle JSON file upload
            if ($request->hasFile('googleCalendarJson')) {
                $file = $request->file('googleCalendarJson');
                $diskName = \App\Services\StorageConfigService::getActiveDisk();
                $fileName = time() . '_' . $file->getClientOriginalName();
                $path = 'google-calendar/' . $fileName;
                
                \Storage::disk($diskName)->put($path, file_get_contents($file));
                $settings['googleCalendarJsonPath'] = $path;
            }

            foreach ($settings as $key => $value) {
                updateSetting($key, is_bool($value) ? ($value ? '1' : '0') : $value);
            }

            return redirect()->back()->with('success', __('Google Calendar settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update Google Calendar settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Update the Google Wallet settings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateGoogleWallet(Request $request)
    {
        try {
            $validated = $request->validate([
                'googleWalletIssuerId' => 'nullable|string|max:255',
                'googleWalletJson' => 'nullable|file|mimes:json|max:2048',
            ]);

            $settings = [
                'googleWalletIssuerId' => $validated['googleWalletIssuerId'] ?? '',
            ];

            // Handle JSON file upload
            if ($request->hasFile('googleWalletJson')) {
                $file = $request->file('googleWalletJson');
                $diskName = StorageConfigService::getActiveDisk();
                $fileName = time() . '_' . $file->getClientOriginalName();
                $path = 'google-wallet/' . $fileName;
                
                \Storage::disk($diskName)->put($path, file_get_contents($file));
                $settings['googleWalletJsonPath'] = $path;
            }

            foreach ($settings as $key => $value) {
                updateSetting($key, $value);
            }

            return redirect()->back()->with('success', __('Google Wallet settings updated successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to update Google Wallet settings: :error', ['error' => $e->getMessage()]));
        }
    }

    /**
     * Clear application cache.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function clearCache()
    {
        try {
            \Artisan::call('cache:clear');
            \Artisan::call('route:clear');
            \Artisan::call('view:clear');
            \Artisan::call('optimize:clear');

            return redirect()->back()->with('success', __('Cache cleared successfully.'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', __('Failed to clear cache: :error', ['error' => $e->getMessage()]));
        }
    }
}   